import java.util.*;

public abstract class PrefixFreeEncoder{
    public char[] indexToChar; // each index contains a different character in the message
                               // Throughout the code we reference the character using its index i.
    public int[] frequencies;  // index i contains the number of times indexToChar[i] appeared in the message used to encode
    public String[] encodingTable; // index i contains the encoding of the character at indexToChar[i] 


    // Constructs the encoder using a specific message
    public PrefixFreeEncoder(String message){
        calcFrequencies(message); // constructs indexToChar and frequencies for that message
        encodingTable = new String[frequencies.length]; // creates the encodingTable array
        buildTable(); // populates the encodingTable array
    }

    // Constructs the encoder using a predefined array of frequencies and corresponding characters
    public PrefixFreeEncoder(int[] frequencies, char[] indexToChar){
        this.frequencies = frequencies;
        this.indexToChar = indexToChar;
        encodingTable = new String[frequencies.length];
        buildTable();
    }

    // returns the index associated with the given character
    public int charIndex(char c){
        for(int i = 0; i < indexToChar.length; i++){
            if(c == indexToChar[i]){
                return i;
            }
        }
        return -1;
    }

    // use the codeword table to encode a message
    public String encode(String message){
        String encoding = "";
        for(int i = 0; i < message.length(); i++){
            encoding += encodingTable[charIndex(message.charAt(i))];
        }
        return encoding;
    }

    // use the codeword table to decode the codeword given
    public String decode(String encoding){
        String message = "";
        String prefix = "";
        for(int i = 0; i < encoding.length(); i++){
            prefix += encoding.charAt(i);
            Character c = getCodeWord(prefix); 
            if(c != null){
                message += c;
                prefix = "";
            }
        }
        if(prefix != ""){
            throw new IllegalArgumentException("encoding given did not decode properly");
        }
        return message;
    }

    // use the frequencies array to make a codeword table
    public abstract void buildTable();

    // if the code given is a valid codeword, return the character that it encodes
    // Otherwise return null
    private Character getCodeWord(String code){
        for(int i = 0; i < encodingTable.length; i++){
            if(encodingTable[i].equals(code)){
                return indexToChar[i];
            }
        }
        return null;
    }

    // populate the frequencies array with the
    // characters contained in the given message
    private void calcFrequencies(String message){
        Map<Character,Integer> charCounts = new TreeMap<>();
        for(int i = 0; i < message.length(); i++){
            char c = message.charAt(i);
            if(charCounts.containsKey(c)){
                charCounts.put(c, charCounts.get(c)+1);
            } else{
                charCounts.put(c, 1);
            }
        }
        this.frequencies = new int[charCounts.size()];
        this.indexToChar = new char[charCounts.size()];
        int i = 0;
        for(char c : charCounts.keySet()){
            indexToChar[i] = c;
            this.frequencies[i] = charCounts.get(c);
            i++;
        }
    }

    // for debugging purposes, prints out the encoding for each character
    public void printEncodingTable(){
        for(int i = 0; i < frequencies.length; i++){
            System.out.println("'" + indexToChar[i] + "':" + encodingTable[i]);
        }
    }

    // Checks that the encoding table has the prefix-free property
    // An encoding is prefix-free if no code words are prefixes of
    // other code words (or equivalently, no code word has another
    // codeword as a prefix).
    // Used for debugging and testing
    public boolean isPrefixFree(){
        for(int i = 0; i < encodingTable.length; i++){
            String code1 = encodingTable[i];
            for(int j = i+1; j < encodingTable.length; j++){
                String code2 = encodingTable[j];
                boolean isprefix = false;
                if(code1.length() <= code2.length()){
                    isprefix = code2.substring(0, code1.length()).equals(code1);
                }
                if(code2.length() <= code1.length()){
                    isprefix = isprefix || code1.substring(0, code2.length()).equals(code2);
                }
                if(isprefix){
                    return false;
                }
            }
        }
        return true;
    }
}