// Parallel version using Java Streams (parallel() split)
//
// To compile and run:
//   javac Stream.java
//   java Stream

import java.util.stream.IntStream;

public class Stream {
    private static final int SIZE = 500_000_000;

    // out[i] = (in[i - 1] + in[i] + in[i + 1]) / 3
    public static void smooth(float[] in, float[] out) {
        int n = in.length;

        // Left boundary
        out[0] = (in[0] + in[1] + in[2]) / 3f;

        // Parallel map for the main body: indices 1 .. n-2
        IntStream.range(1, n - 1)
                .parallel()
                .forEach(i -> out[i] = (in[i - 1] + in[i] + in[i + 1]) / 3f);

        // Right boundary
        out[n - 1] = (in[n - 3] + in[n - 2] + in[n - 1]) / 3f;
    }

    public static void main(String[] args) {

        // Create some data
        float[] array = generateData(SIZE);

        // Run the test a few times to warm up the JVM
        for (int i = 0; i < 5; i++) {
            runTest(array);
        }
    }

    private static void runTest(float[] array) {
        float[] smoothed = new float[array.length];

        long startTime = System.nanoTime();
        smooth(array, smoothed);
        long endTime = System.nanoTime();
        double elapsedTimeMS = (endTime - startTime) / 1_000_000.0;

        int checksum = 17;
        for (float v : smoothed) {
            checksum = checksum * 31 + (int)v;
        }

        System.out.printf("Stream Parallel: Time taken: %.3f ms, checksum: %d%n",
                elapsedTimeMS, checksum);
    }

    // Utility method to generate "random" float data
    private static float[] generateData(int size) {
        float[] data = new float[size];
        for (int i = 0; i < size; i++) {
            data[i] = (float) i;
        }
        return data;
    }
}