import itertools
from PIL import Image
import sys


def csv_line_to_pixels(line):
    """
    Given a CSV-formatted row of integers, returns the integers
    as a list. The argument line must be a string such as

        "255, 0,  27"

    Note that
     * There are no extra spaces at the beginning or end of
       the string, and
     * Commas are only between elements (no comma before the
       first element, and no comma after the last one).

    This method will present an error if either of these
    constraints are violated.
    """
    row = []
    for pixel in line.split(','):
        row.append(int(pixel))
    return row


def convert_to_grayscale(input_file, out_file):
    try:
        in_image = Image.open(input_file)
    except IOError as e:
        print(e)
        sys.exit(1)
    except Exception:
        print("Unexpected error reading file", input_file)
        sys.exit(1)

    out_image = in_image.convert("L")

    try:
        # Write the image
        out_image.save(out_file, "PNG")
    except IOError as e:
        print(e)
    except Exception:
        print("Unexpected error writing file", out_file)


def read_image(file_path):
    """
    Reads the image file at file_path into a rectangular grid of pixels,
    represented as a list of list of integer. Each element of the outer list
    is one row of pixels, where each pixel is an integer x such that
    0 <= x < 256. Returns the grid of pixels.
    """
    # YOU DO NOT NEED TO MODIFY THIS FUNCTION.
    # Do not worry about understanding "How" this function works.
    print("Reading image", file_path)

    # Open a file in image format
    try:
        image_file = Image.open(file_path)
    except IOError as e:
        print(e)
        return
    except Exception:
        print("Unexpected error reading file", file_path)
        return

    width, height = image_file.size
    data = list(image_file.getdata())

    # data is a single list. We break it into a nested list.
    result = []
    for r in range(height):
        # Get the part of the list that corresponds to this row.
        row_start = r * width
        row = data[row_start:row_start + width]

        result.append(row)
    return result


def write_image(file_name, pixel_grid):
    """
    Given pixel_grid as an image in a list of lists of integers format,
    write it to the filename file_name as an image.
    Requires that:
    * Each row of pixel_grid is of the same length
    * Each pixel value is an integer x such that 0 <= x < 256.
    """
    # YOU DO NOT NEED TO MODIFY THIS FUNCTION.
    # Do not worry about understanding "How" this function works.
    size = len(pixel_grid[0]), len(pixel_grid)
    image = Image.new("L", size)

    print("Writing", size[0], 'x', size[1], "image to file", file_name)

    # Flatten the list by making an iterable sequence over the inner lists
    # and then materializing the whole list.
    data = list(itertools.chain.from_iterable(pixel_grid))
    image.putdata(data)

    try:
        # Write the image. File extension of file_name determines
        # the encoding.
        image.save(file_name)
    except IOError as e:
        print(e)
    except Exception:
        print("Unexpected error writing file", file_name)


def write_grid(file_name, pixel_grid):
    """
    Writes the given pixel_grid to filename file_name as CSV.
    """
    # YOU DO NOT NEED TO MODIFY THIS FUNCTION.
    # Do not worry about understanding "How" this function works.
    output_file = open(file_name, 'w')

    for row in pixel_grid:
        output_file.write(str(row[0]))
        for column in range(1, len(row)):
            output_file.write(', ' + str(row[column]).rjust(3))
        output_file.write('\n')

    output_file.close()


def parse_args():
    # Print how to use the program correctly if it appears that it has
    # been used incorrectly (with the wrong number of arguments).
    if len(sys.argv) != 2:
        print("Usage:", sys.argv[0], "<input_file>")
        print("  <input_file> should be either ")
        print("    (a) a CSV-formatted text file")
        print("       (as generated by the write_grid function), or")
        print("    (b) a black-and-white image file")
        print("       (as generated by color_to_gray.py).")
        print()
        print("  Blurs the given input file and outputs the result as:")
        print("   <input_file>_blurry.png (an image) and")
        print("   <input_file>_blurry_grid.txt (a CSV-formatted text file).")
        sys.exit()

    return sys.argv[1]
