import java.awt.*;
import javax.swing.*;
import java.awt.geom.*;
import java.util.*;

public class TreePanel extends JPanel {
    private TreeNode root;
    private JTextArea text;
    private TreeNode pinkNode;
    private int delay;
    private boolean commaBefore;

    public static final int XGAP = 120;
    public static final int YGAP = 40;

    public TreePanel(JTextArea text, int delay) {
        root = null;
        this.text = text;
        this.delay = delay;
        setBackground(Color.CYAN);
    }

    public void clear() {
        root = null;
        repaint();
    }

    public void add(String s) {
        root = add(s, root, 0, 1);
        fixClashes(root);
        pinkNode = null;
        repaint();
    }

    private void scoot(TreeNode root, double amount) {
        if (root != null) {
            root.x += amount;
            scoot(root.left, amount);
            scoot(root.right, amount);
        }
    }

    private void fixPair(TreeNode root, TreeNode node1, TreeNode node2) {
        if (node1 != null && node2 != null) {
            double difference = node2.x - node1.x;
            if (difference < 1) {
                int steps = 50;
                for (int i = 1; i <= steps; i++) {
                    scoot(root.left, (difference - 1)/2/steps);
                    scoot(root.right, (1 - difference)/2/steps);
                    repaint();
                    RepaintManager.currentManager(text).paintDirtyRegions();
                }
            }
            fixPair(root, node1.left, node2.left);
            fixPair(root, node1.left, node2.right);
            fixPair(root, node1.right, node2.left);
            fixPair(root, node1.right, node2.right);
        }
    }

    private void fixClashes(TreeNode root) {
        if (root != null) {
            fixClashes(root.left);
            fixClashes(root.right);
            fixPair(root, root.left, root.right);
        }
    }

    private TreeNode add(String next, TreeNode node, double x, double y) {
        if (node == null)
            node = new TreeNode(next, x, y);
        else {
            displayData(node, false);
            if (node.data.compareTo(next) >= 0)
                node.left = add(next, node.left, node.x - 0.5, y + 1);
            else
                node.right = add(next, node.right, node.x + 0.5, y + 1);
        }
        return node;
    }

    public void preorderTraversal() {
        text.setText(" ");
        commaBefore = false;
        preorderTraversal(root);
        pinkNode = null;
        repaint();
    }

    public void setDelay(int delay) {
        this.delay = delay;
    }

    private void preorderTraversal(TreeNode root) {
        if (root != null) {
            displayData(root, true);
            preorderTraversal(root.left);
            preorderTraversal(root.right);
        }
    }

    private void displayData(TreeNode root, boolean inTextArea) {
        if (inTextArea) {
            if (commaBefore)
                text.append(", ");
            commaBefore = true;
            text.append(root.data);
            RepaintManager.currentManager(text).paintDirtyRegions();
        }
        pinkNode = root;
        repaint();
        RepaintManager.currentManager(this).paintDirtyRegions();
        pause();
    }

    public void inorderTraversal() {
        text.setText(" ");
        commaBefore = false;
        inorderTraversal(root);
        pinkNode = null;
        repaint();
    }

    private void inorderTraversal(TreeNode root) {
        if (root != null) {
            inorderTraversal(root.left);
            displayData(root, true);
            inorderTraversal(root.right);
        }
    }

    public void postorderTraversal() {
        text.setText(" ");
        commaBefore = false;
        postorderTraversal(root);
        pinkNode = null;
        repaint();
    }

    private void postorderTraversal(TreeNode root) {
        if (root != null) {
            postorderTraversal(root.left);
            postorderTraversal(root.right);
            displayData(root, true);
        }
    }

    private void paintNode(Graphics2D g2, TreeNode node, Color c) {
        double x = node.x * XGAP;
        double y = node.y * YGAP;
        int fontHeight = g2.getFont().getSize();
        int ovalHeight = fontHeight + 2;
        g2.setColor(c);
        g2.fill(new RoundRectangle2D.Double(x - XGAP /3.0, y - ovalHeight/2, 
                                            XGAP * 2.0/3.0, ovalHeight, 10, 10));
        g2.setColor(Color.BLUE);
        double textWidth = g2.getFontMetrics().stringWidth(node.data);
        g2.drawString(node.data, Math.round(x - textWidth/2), 
                      Math.round(y + fontHeight/2));
    }

    private void paint(Graphics2D g2, TreeNode root) {
        if (root != null) {
            drawLine(g2, root, root.left);
            drawLine(g2, root, root.right);
            if (root == pinkNode)
                paintNode(g2, root, Color.PINK);
            else
                paintNode(g2, root, Color.WHITE);
            paint(g2, root.left);
            paint(g2, root.right);
        }
    }

    private void drawLine(Graphics2D g2, TreeNode node1, TreeNode node2) {
        g2.setColor(Color.BLACK);
        if (node1 != null && node2 != null) {
            g2.draw(new Line2D.Double(node1.x * XGAP, node1.y * YGAP,
                                      node2.x * XGAP, node2.y * YGAP));
        }
    }

    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D g2 = (Graphics2D)g;
        g2.translate(getWidth()/2, 0);
        paint(g2, root);
    }

    private void pause() {
        try {
            Thread.sleep(delay);
        } catch (Exception e) {
            throw new InternalError();
        }
    }
}
